% Example for creating and using a StateSpaceModel, based on 
% Van Straten et al, Optimal Control of Greenhouse Cultivation (2010), Chapter 2 [1]. 
% It is recommended to run this example section by section by using
% 'Run and Advance', and at each step to view what happened.

% David Katzin, Wageningen University
% david.katzin@wur.nl

%% Create a new StateSpaceModel object
m = StateSpaceModel();

%% Define the time span
m.t = DynamicElement('01/01/2001 00:00:00', [], '01/01/2001 00:00:00', []);
m.t.val = [0 48]; % hours

%% Define parameters
addParam(m, 'lue', 7.5e-8);
addParam(m, 'heatLoss', 1);
addParam(m, 'heatEff', 0.1);
addParam(m, 'gasPrice', 4.55e-4);
addParam(m, 'lettucePrice', 136.4);
addParam(m, 'heatMin', 0);
addParam(m, 'heatMax', 100);

%% Define inputs 
m.d.rad = DynamicElement('d.rad');
m.d.tempOut = DynamicElement('d.tempOut');

%% Define states and controls
m.x.dryWeight = DynamicElement('x.dryWeight');
m.x.tempIn = DynamicElement('x.tempIn');
m.u.heating = DynamicElement('u.heating');

%% Define auxiliary states
% Photosynthesis [kg m^{-2} h^{-1}], equation 2.1 [1] 
m.a.phot = m.p.lue.*m.d.rad.*m.x.tempIn.*m.x.dryWeight; 

% Heat loss to the outside [degC h^{-1}], equation 2.2 [1] 
m.a.heatOut = m.p.heatLoss.*(m.d.tempOut - m.x.tempIn); 

% Heat gain from the pipes [degC h^{-1}], equation 2.2 [1] 
m.a.heatIn = m.p.heatEff.*m.u.heating;

%% Set the ODESs
% Photosynthesis [kg m^{-2} h^{-1}], equation 2.1 [1] 
m.x.dryWeight = DynamicElement(m.a.phot); 
% Notice that we used DynamicElement(m.a.phot). 
% This is because simply stating `m.x.dryWeight = m.a.phot` would cause 
% `x.dryWeight` to be equivalent to `a.phot`, and any change to the one would 
% influence the other (recall handle classes). 

% Heat gain in the greenhouse [degC h^{-1}], equation 2.2 [1] 
m.x.tempIn = m.a.heatOut + m.a.heatIn;

%% Set controls (as inputs)
time = (0:48)';
m.u.heating.val = [time zeros(size(time))];

%% Set the values of the inputs
m.d.rad.val = [time max(0, 800*sin(4*pi*time/48-0.65*pi))];
m.d.tempOut.val = [time 15+10*sin(4*pi*time/48-0.65*pi)];

%% Set initial values for the states
m.x.dryWeight.val = 1; % kg m^{-2} 
m.x.tempIn.val = 10; % degC

%% Make some plots
% figure; plot(m.d.rad);
% figure; plot(m.d.tempOut);
% figure; plot(m.u.heating);
% plot(m);

%% Simulate
runSim(m,1); 
% alternatively, use 
% [t,x] = ode15s(@(t,x) getOdes(m,t,x),m.t.val,getInitialStates(m))
plot(m);

%% Rule based control

% make a copy of m
ruleBased = StateSpaceModel(m);

% bang-bang cotrol
ruleBased.u.heating = ifElse('x.tempIn < 15', ruleBased.p.heatMax.val, ruleBased.p.heatMin.val); 
runSim(ruleBased,1);


figure;
subplot(2,1,1);
plot(m.x.tempIn); grid; hold on;
plot(ruleBased.x.tempIn);
legend('tempIn no heating','tempIn bang bang heating');

subplot(2,1,2);
plot(ruleBased.u.heating); hold on
legend('bang bang heating');

mProfit = m.p.lettucePrice.val*m.x.dryWeight.val(end,2)-m.p.gasPrice.val*trapz(m.u.heating);
rbProfit = ruleBased.p.lettucePrice.val*ruleBased.x.dryWeight.val(end,2)-ruleBased.p.gasPrice.val*trapz(ruleBased.u.heating);
fprintf('\nProfit with no heating: %.3f\n',mProfit);
fprintf('Profit with bang bang control: %.3f\n',rbProfit);
%% higher resolution

% make a copy of ruleBased
hiRes = StateSpaceModel(ruleBased);

runSim(hiRes,0.1);
subplot(2,1,1); 
plot(hiRes.x.tempIn);
legend('tempIn no heating','tempIn bang bang heating','tempIn bang bang hi res');

subplot(2,1,2); 
plot(hiRes.u.heating);
legend('bang bang heating', 'bang bang heating hi res');

hrProfit = hiRes.p.lettucePrice.val*hiRes.x.dryWeight.val(end,2)-hiRes.p.gasPrice.val*trapz(hiRes.u.heating);
fprintf('\nProfit with no heating: %.3f\n',mProfit);
fprintf('Profit with bang bang control: %.3f\n',rbProfit);
fprintf('Profit with hi-res bang bang control: %.3f\n',hrProfit);

%% proportional control

propCont = StateSpaceModel(hiRes);
propCont.x.tempIn.def = 'x.tempIn'; % give x.tempIn original definition
propCont.u.heating = proportionalControl(propCont.x.tempIn, 17, -4, propCont.p.heatMin.val, propCont.p.heatMax.val);

propCont.x.tempIn = propCont.a.heatOut + propCont.a.heatIn; % give x.tempIn ODE

runSim(propCont,0.1);

subplot(2,1,1); 
plot(propCont.x.tempIn);
legend('tempIn no heating','tempIn bang bang heating',...
    'tempIn bang bang hi res', 'tempIn proportional heating');
subplot(2,1,2); 
plot(propCont.u.heating);
legend('bang bang heating', 'bang bang heating hi res',...
    'proportional heating');

pcProfit = propCont.p.lettucePrice.val*propCont.x.dryWeight.val(end,2)-propCont.p.gasPrice.val*trapz(propCont.u.heating);
fprintf('\nProfit with no heating: %.3f\n',mProfit);
fprintf('Profit with bang bang control: %.3f\n',rbProfit);
fprintf('Profit with hi-res bang bang control: %.3f\n',hrProfit);
fprintf('Profit with proportional control: %.3f\n',pcProfit);

%% Optimal control using Tomlab

% define constraints, using Tomlab syntax
mTom = StateSpaceModel(m);

mTom.c.heating = 'p.heatMin <= icollocate(u.heating) <= p.heatMax';
    
% define goal, using Tomlab syntax
mTom.g = '-p.lettucePrice*final(x.dryWeight)+integrate(p.gasPrice*u.heating)';
    
% solve using Tomalb
solveTomlab(mTom, 600, []);

subplot(2,1,1); 
plot(mTom.x.tempIn);
legend('tempIn no heating','tempIn bang bang heating',...
    'tempIn bang bang hi res', 'tempIn proportional heating','tempIn Tomlab');

subplot(2,1,2); 
plot(mTom.u.heating);

legend('bang bang heating', 'bang bang heating hi res',...
    'proportional heating', 'tomLab heating');

tlProfit = mTom.p.lettucePrice.val*mTom.x.dryWeight.val(end,2)-mTom.p.gasPrice.val*trapz(mTom.u.heating);
fprintf('\nProfit with no heating: %.3f\n',mProfit);
fprintf('Profit with bang bang control: %.3f\n',rbProfit);
fprintf('Profit with hi-res bang bang control: %.3f\n',hrProfit);
fprintf('Profit with proportional control: %.3f\n',pcProfit);
fprintf('Profit with optimal control: %.3f\n',tlProfit);

%% Adjust plot
axis([0 50 0 100]);